/* A decision procedure for well-formed linear quantum cellular automata
   - Christoph Durr, Huong Le Thanh, Miklos Santha
   uses LEDA aviable at ftp://ftp.mpi-sb.mpg.de
 */

#include "my_d_array.h"
#include <LEDA/string.h>
#include <LEDA/graph.h>
#include <LEDA/graph_alg.h>
#include <LEDA/set.h>
#include <LEDA/stream.h>
#include <LEDA/array.h>
#include <Complex.h>      // Complex numbers are taken from from libg++
#include <assert.h>
#include "bf.h"

#ifndef PI
#define PI 3.1415927
#endif
                          // Only some little global variables. 
                          // Please forgive me, dear algo-teacher!
double precision       = 0.001;
bool   verify          = true;
bool   print_graphs    = false;
bool   print_automaton = false;
bool   print_bitmap    = false;
bool   print_bold      = false;
const  Complex zero(0);

// A state is his character, used to print or read
typedef char                   state;

// -------------------------- PAIR --------------------------------
/* Generic class pair, used for the second part of the algorithm */
template <class S>
class pair {
public:
  pair(void) {}
  pair(const S &i1, const S &i2) :s1(i1), s2(i2) {}
  bool diagonal(void) const {return s1==s2;}
  const S &first(void)  {return s1;}  
  const S &second(void) {return s2;}
  friend int compare(const pair<S> &C, const pair<S> &D); 
  bool operator==(const pair<S>&p) const {return s1==p.s1 && s2==p.s2;}
  bool operator<=(const pair<S>&p) const {
    return s1<p.s1 || s1==p.s1 && s2<=p.s2;
  }
private:
  S s1,s2;
};

typedef pair<string> pair_string;
typedef pair<node>   pair_node;

                            // It would be nice to write "const pair<S>"
                            // But then we get an overload fct pb with
                            // predefined fcts of LEDA.
template <class S>
void Print(pair<S> &p, ostream &out) {out<<p.first()<<' '<<p.second();}

template <class S>
int compare(pair<S> &C, pair<S> &D) 
{
  if (C == D) return 0;
  if (C <= D) return -1;
  return +1;
}

// -------------------------- SET ---------------------------------------
typedef set<state> state_set;

/* Reads a state set of the form {0,1,2}
 */
istream &operator>>(istream &in, set<state> &A)
{
  char  sep;
  state q;
  in >> sep;
  assert( sep=='{' );
  sep = ',';         // This is a trick to get sep always ','
  while (sep != '}' ) {
    assert( sep== ',');
    in >> q;
    assert(q!='.');
    A.insert(q);
    in >> sep;
  }
  return in;
}

/* Prints a state set in the above form
 */
ostream &operator<<(ostream &out, const set<state> &A)
{
  state q;
  bool first=true;
  out << "{";
  forall(q,A) {
    if (first) 
      first = false;
    else 
      out << ",";
    out << q;
  }
  return out << "}";
}

// -------------------------- COMPLEX -----------------------------------
const char *dtoa(double d)
{
  static char buf[80];
  sprintf(buf, "%g", d);
  return buf;
}

void Print(ostream &out, const Complex &c) {
  if (imag(c)==0)
    out << dtoa(real(c));
  else
    if (real(c)==0)
      out << dtoa(imag(c)) << "i";
  else
    out << "(" << dtoa(real(c)) << "+" << dtoa(imag(c)) << "i)";
}

// -------------------------- SUPERPOSITION -----------------------------
/* Superpositions of classical set E 
   u * v is inner product
   u | v is true if u is orthogonal to v
   norm() returns norm L_2
   */

#define forall_nonzero(e,u) forall_defined(e,u.ampl) 

template <class E>
class superpos {
public:
  superpos() :ampl(zero) {}
  superpos(E e, Complex a=1) 
    :ampl(zero) {ampl[e] = a;}    // basis vector indexed by e
  const Complex &operator[](const E &e) const {
    if (ampl.defined(e))
      return ampl[e];
    else
      return zero;
  }
  superpos<E> &operator=(const superpos<E> &u) {
    E e;
    empty();
    forall_nonzero(e,u)
      if (abs(u[e]) >= precision)
	ampl[e] = u[e];
  }
  superpos<E> &operator+=(const superpos<E> &u) {
    E e;
    forall_nonzero(e,u)
      ampl[e] += u[e];
  }
  Complex &operator[](const E &e) {
      return ampl[e];
  }
  Complex operator*(const superpos<E> u) const {
    Complex retval = 0;
    E e;
    forall_nonzero(e, u)
      retval += (*this)[e] * conj( u[e] );
    return retval;
  }
  bool operator|(const superpos<E> u) const {
    return abs((*this) * u) < precision;
  }
  double norm() const {
    double retval = 0;
    E e;
    forall_nonzero(e, (*this) )
      retval += ::norm( (*this)[e] ); // This is |(*this)[e]|^2
    return sqrt(retval);
  }
  void empty(void) {
    E e;                   // don't use forall_nonzero, avoid the precision pb
    forall_defined(e,ampl)
      ampl.undefine(e);
  }
  bool is_empty(void) const {
    return ampl.empty();
  }
  ostream &print(ostream &out, bool tabular = false) const {
    E e;
    bool first=true;
    forall_nonzero(e, (*this)) {
      if (first)
	first = false;
      else {
	out << " + ";
	if (tabular) out << endl;
      }
      Print(out, (*this)[e]);
      if (tabular) out << "\t";
      out << "|" << e << ">";
    }
    return out;
  }
  d_array<E,Complex> ampl;
};

template <class E>
const superpos<E> &operator*(Complex fact, const superpos<E> &u) {
  static superpos<E> retval;
  E e;
  retval.empty();
  forall_nonzero(e,u)
    retval[e] = fact * u[e];
  return retval;
}

// -------------------------- LQCA -----------------------------
// superpostion of states must be defined as a type because
// in g++ one can not write "d_array<string, superpos<state>>"
typedef superpos<state>            stateSupo;
typedef d_array<string, stateSupo> localFct;

class Lqca {
public:
  Lqca(void) {}
  Lqca(const state_set &S, state q, int b, int e, const localFct &l) 
    :states(S), quiescent(q), beg(b), end(e), local(l) {}
  state_set  states;
  state      quiescent;
  int        beg, end;                  // neighborhood [beg..end]
  unsigned   span() const {return end-beg+1;}
  localFct   local;
  ostream &print(ostream &out) const {
    string n;
    out << states << " " << quiescent << endl;
    out << "[" << beg << ".." << end << "]\n{\n";
    forall_defined(n,local) {
      out << "  local(" << n << ")= ";
      local[n].print(out) << endl;
    }
    return out << "}\n";
  }
};

/* Sets the local transition amplitudes replacing '.' by 
   all states
   */
void set_local(Lqca &A, const d_array<char, state_set> &var,
	       string N, Complex ampl)
{
  state x;
  char  wildcard = 0;
  int pos = N.pos('.');
                              // 1) replace all . by every state
  if (pos != -1) {
    forall(x, A.states) {
      N[pos] = x;
      set_local(A, var, N, ampl);
    }
    return;
  }
                              // 2) search for variables
  for(pos=0; pos<N.length(); pos++)
    if (var.defined(N[pos])) {
      wildcard = N[pos];
      break;
    }
    else assert( A.states.member(N[pos]) );

  if (wildcard)               // 3) replace variables
    forall(x, var[wildcard])
      set_local(A, var, N.replace_all(wildcard, x) ,ampl);
  else {
                              // 4) check multiple definitions
#if 0
    if (A.local[N][q] != 0)
      cout << "Warning: Overwriting previous definition of transition "
	<<N<<" "<<q<<endl;
#endif
    A.local[ N.tail(A.span()) ][ N[0] ] = ampl;
  }
}
      
/* Reads the local function in the form
   { 000 0 -1 
     ...
     001 1 0.707 }
   */
void Read_local(Lqca &A, istream &in)
{
  string   N;
  state    q;
  Complex  ampl;
  d_array<char, state_set> var;

  in >> N;
  while (N != "{") {
    state_set states, wild;
    state x;
    assert( N == "forall" );
    in >> wild;
    in >> N; assert( N=="in" );
    in >> states;
    forall(x, wild)
      var[x] = states;
    in >> N;
  }
  assert( N=='{' );
  in >> N;
  while (N != '}') {
    assert (N.length() == A.span());
    in >> q;
    in >> ampl;
    set_local(A, var, q+N, ampl);
    in >> N;
  }
}

/* The neighborhood is of form 34 or [-2..1] */
void Load(Lqca &A, const string &filename) 
{
  char buf[80];
  unsigned i;
  file_istream source(filename);
  source >> A.states;
  source >> A.quiescent;    
  source >> buf;
  if (1==sscanf(buf,"%u",&i)) {
    A.beg=0;
    A.end=i-1;
  }
  else {
    i = sscanf(buf,"[%d..%d]", &A.beg, &A.end);
    assert(i==2);
    assert(A.beg<=A.end);
  } 
  Read_local(A, source);
  if (print_automaton)
    A.print(cout);
}

// -------------------------- CONFIGURATION ---------------------
/* Configuration 
   c || x returns a configuration where x is appended 
          at the rightmost cell of c
   config(q, s,i) creates a config with s_0..s_k at cell i..i+k
   */
class config {
public:
  config(state q=0) :quiescent(q) {}
  config(state q, const array<state> &t) :quiescent(q), nontriv(t) {}
  config(const config &c) :quiescent(c.quiescent), nontriv(c.nontriv) {}
  config(state q, const string &s, int pos=0) 
    :quiescent(q), nontriv(pos,pos+s.length()-(s==""? 0:1)) {
      if (s == "") 
	nontriv[pos] = q;
      else
	for (int i=0; i<s.length(); i++)
	  nontriv[i+pos] = s[i];
    }
  state operator[](int i) const {
    if (nontriv.low() <= i && i <= nontriv.high())
      return nontriv[i];
    else
      return quiescent;
  }
  config operator || (state x) const {
    int i;
    array<state> tab(nontriv.low(), nontriv.high()+1);
    for (i=nontriv.low(); i<=nontriv.high(); i++)
      tab[i] = nontriv[i];
    tab[i] = x;
    return config(quiescent, tab);
  }
  int beg(void) const {return nontriv.low();}
  int end(void) const {return nontriv.high();}
friend ostream &operator<<(ostream &out, const config &c);
private:
  state quiescent;
  array<state> nontriv;
};

/* define an arbitrary order on configurations */
int compare(const config &a, const config &b) {
  int retval;
  for(int i= Min(a.beg(), b.beg()); 
      i   <= Max(a.end(), b.end()); i++) {
    retval = compare(a[i], b[i]);
    if (retval)
      return retval;
  }
  return retval;
}

ostream &operator<<(ostream &out, const config &c) {
  out << ".." << c.quiescent;
  for (int i=c.nontriv.low(); i<=c.nontriv.high(); i++)
    if (print_bold && i==0)
      out << "{\\bf " << c.nontriv[i] << "}";
    else
      out << c.nontriv[i];
  out << c.quiescent << "..";
  return out;
}    

/* return the neighborhood of cell pos */  
string neighbor(const Lqca &A, const config&c, int pos) {
  string retval;
  for(int a=A.beg; a<=A.end; a++)
    retval += c[pos+a];
  return retval;
}

// -------------------------- TIME EVOLUTION OPERATOR ----------------
/* Builds the tensor product
 */
const superpos<config> &tensor(state q,             // quiescent state
			       const superpos<config> &u,
			       const superpos<state> &v, 
			       int pos) 
{
  state  x;
  static superpos<config> retval;
  retval.empty();
  if (u.is_empty())
    forall_nonzero(x,v) {
      config c(q, x, pos);
      retval[ c ] += v[x];
    }
  else {
    config c;
    forall_nonzero(c,u)
      forall_nonzero(x,v) {
	retval[c || x] += u[c]*v[x];
      }
  }
  return retval;
}

/* returns the column vector of the global operator indexed by c
 */
superpos<config> column(const Lqca &A, const config &c)
{
  superpos<config> retval;
  for (int i=c.beg()-A.end; i<=c.end()-A.beg; i++)
    retval = tensor(A.quiescent, retval, A.local[ neighbor(A,c,i)], i);
  return retval;
}

/* returns the entry Delta(d,c), i.e. the transition amplitude to
   go from d to c.
   */
Complex global(const Lqca &A, const config &d, const config &c)
{
  Complex retval = 1;
  for (int i=Min(d.beg(), c.beg()); i<=Max(d.end(), c.end()); i++)
    retval *= A.local[ neighbor(A,c,i) ][ d[i] ];
  return retval;
}

// -------------------------- ORDER CHARSET --------------------------
/* returns a string which associates to each integer 0..|Sigma|-1
   a distinct state, thus returning an arbitrary order.
   */
string ordStates(const set<state> &Sigma) {
  string S;
  state  q;
  forall(q,Sigma) S += q;
  return S;
}

/* Returns for each 'order' 0..|Sigma|^length-1
   a distinct string of length 'length',
   thus ordering the words of this same length
   */
string ordWords(const string   &states,
		unsigned length,
		unsigned order)
{
  string   S;
  unsigned card = states.length();
  for (int i=0; i<length; i++) {
    S     += states[order % card];
    order /= card;
  }
  return S;
}

string adj_node(const string &u, state q) {
  return u.tail(u.length()-1) + q;
}

// -------------------------- NORMAL COLUMNS ----------------------------
/* Creates a de Bruijn Graph
   weighted by the norm of the associated local transition fct
   return the node associated to the quiescent state word
 */
node deBruijn(GRAPH<string,string> &G,
	      edge_array<double> &W,
	      const Lqca &A)
{
  string   Sigma  = ordStates(A.states);
  unsigned S_card = Sigma.length(); 
  unsigned N_card = (unsigned)pow((double)S_card, (int)(A.span()-1));
  d_array<string,node> N;
  string   u,v;
  edge     e;
  state    q;

  W.init(G, S_card * N_card, 0);
                                  // create nodes
  for (int i=0; i<N_card; i++) {
    u      = ordWords(Sigma, A.span()-1, i);
    N[ u ] = G.new_node(u);
  }
                                  // create edges
  forall_defined(u,N) {
    forall(q, A.states) {
      v = adj_node(u,q);
      e = G.new_edge(N[u], N[v], u+q);
      W[ e ] = A.local[ u+q ].norm(); // and weights
    }
  }
                    // Build the quiescent state word
  u = "";
  for (int j=0; j<A.span()-1;  j++)
    u += A.quiescent;
  return N[ u ];
}

bool isNormal(GRAPH<string,string> &G, 
	      node &quie,
	      const Lqca &A) 
{
  edge_array<double>   W;
  edge                 e;
                         // building the de Bruijn graph
  quie = deBruijn(G, W, A);
  if (print_graphs)
    G.print();
                         // verifying that for all x, |A.local(x)| != 0
  forall_edges(e,G)
    if (W[e] == 0) {
      cout << "The local transition function associates to '"<<G[e]<<
              "' a superposition of norm zero.\n";
      return false;
    }
                         // check for vectors of norm < 1
  if (bf_less(G, quie, W) == false) {
    cout << "There is a column vector of norm less than 1\n";
    exit(1);
  }
                        // check for vectors of norm > 1
  if (bf_greater(G, quie, W) == false) {
    cout << "There is a column vector of norm greater than 1\n";
    exit(1);
  }
  return true;
}

// -------------------------- ORTHOGONAL COLUMNS -----------------------
bool isOrthogonal(const GRAPH<string,string> &G,
		  node  quie,
		  const Lqca &A) 
{
  GRAPH<pair_string, char> H;
  d_array<pair_node, node> D;
  node u,v;
  edge e,f;
                     // build H, the nodes ...
  forall_nodes(u,G)
    forall_nodes(v,G) {
      pair_string ps(G[u], G[v]);
      pair_node   pn(u,    v);
      D[pn] = H.new_node(ps);
    }
                     // ... and edges.
  forall_edges(e,G)
    forall_edges(f,G) 
      if (!(A.local[G[e]] | A.local[G[f]])) {
	pair_node source(G.source(e), G.source(f));
	pair_node target(G.target(e), G.target(f));
	u = D[ source ];
	v = D[ target ];
	H.new_edge( u, v, 0);
      }
  if (print_graphs)
    H.print();
                      // compute strongly connected componements
  node_array<int> comp(H);
  STRONG_COMPONENTS(H, comp);
                     // i is the number of the str.conn.comp. containing
                     // the node associated to the quiescent state
  pair_node quie2(quie, quie);
  u = D[ quie2 ];
  int i = comp[ u ];
                     // verifying that all nodes in this str.conn.comp.
                     // are diagonal, i.e. of the form (x,x)
  forall_nodes(u,H)
    if (i==comp[u] && !H[u].diagonal() ) {
      cout << "The column vectors are not orthogonal\n";
      return false;
    }
  return true;
}

// ------------------------- ORTHONORMAL ----------------------------
bool wellformed(const Lqca &A) 
{
  node                 quie;
  GRAPH<string,string> G;
  if (!isNormal(G, quie, A)) 
    return false;
  return isOrthogonal(G, quie, A);
}

// -------------------------- SIMULATION ----------------------------
void simulate(const Lqca &A, string initial, int steps)
{
  config c(A.quiescent, initial);
  superpos<config> past(c), future;
  for (int i=0 ;i<=steps; i++) {
    cout << "----------- "<< i << endl;
    past.print(cout, true) << endl; 
                              // We don't need to compute the last step
    if (i < steps) {
      superpos<config> future;
      forall_nonzero(c,past) {
	future += past[c] * column(A, c);
      }
      past = future;          // Indian philosophy
    }
  }
}

// -------------------------- GLOBAL MATRIX -------------------------
/* We define the following bijection between N(pos. int) and C (finite conf.)
   Let c be a config, nontrivial in [-k,k]
   Let w(c)=c(0) c(1) c(-1) c(2) c(-2) .. c(k) c(-k) be a finite string

            { 0                       if s is the empty string
   and f(s)={ 
            { ord(x) + |Sigma| * f(u) if w=xu for some string u, and letter x

   Given an integer i this function returns the config. c s.t. f(w(c))=i
   it builds c state by state in a string w. idx indicates the last insertion
   point and in this loop will take te values 0, 1, -1, 2, -2, ...
   min will be the minimal idx encountered.
   */
config ordConfig(const Lqca &A, unsigned i) 
{
  string ord = ordStates(A.states);
  int card   = ord.length();
  int min    = 0;
  int idx    = 0;
  string w   = "";
  state  x;
  while (i>0) {
    x = ord[i % card];
    i /= card;
    if (idx<min) 
      min = idx;
    if (idx<=0) {
      idx = -idx+1;
      w = x+w;
    } 
    else {
      idx = -idx;
      w = w+x;
    }
  }
  return config(A.quiescent, w, min);
}

/* bitPixel est utilise pour l'affichage en bitmap
 */
inline char bitPixel(bool f) {return f? '#': '-';}

/* Prints the global matrix restricted to the first n columns and lines
   */
ostream &PrintGlobal(const Lqca &A, unsigned n, ostream &out)
{
  static const string col_sep="\t&";
  static const string row_sep="\t\\\\\n";
  int col, row;
                           // print the header line
  if (print_bitmap) {
                           // print marks |     |     |
    out << bitPixel(false)  << bitPixel(false);
    for(col=0; col<n; col++) 
          out << bitPixel(col%(A.states.size() * A.states.size())==0);
    out << endl;
                           // print marks | | | | | | |
    out << bitPixel(false)  << bitPixel(false);
    for(col=0; col<n; col++) 
          out << bitPixel(col%A.states.size() == 0);
    out << endl;
    
    for(row=0; row<n; row++) {
      config row_c = ordConfig(A, row);
                           // print marks -   -   -   -
	out << bitPixel(row%(A.states.size() * A.states.size())==0);
                           // print marks - - - - - - -
	out << bitPixel(row%A.states.size() == 0);

      for(col=0; col<n; col++) {
	config col_c = ordConfig(A, col);
	out << bitPixel( abs(global(A, row_c, col_c)) >= precision);
      }
      out << endl;
    }
  }
  else {                   // print a LaTeX table
                           // print columns header
    for(col=0; col<n; col++) {
      config col_c = ordConfig(A, col);
      out << col_sep << col_c;
    }
    for(row=0; row<n; row++) {
      config row_c = ordConfig(A, row);
                           // print row header
      out << row_sep << row_c;
      for(col=0; col<n; col++) {
	config col_c = ordConfig(A, col);
	Print(out << col_sep, column(A, col_c) [row_c]);
      }
    }
    out << row_sep;
  }
  return out;
}

// -------------------------- ENUMERATE -----------------------------
/* enumerate all well-formed QLCA with restricted amplitudes.
   This is not really efficient !
*/
void en2(const state_set &Sigma, int q, const string &ord);
void en3(const state_set &Sigma, int q, const string &ord,
	 localFct local, int x, int y);
void en4(const state_set &Sigma, int q, const string &ord,
	 localFct local, int x, int y,
	 state_set touch, int next);
void en5(const state_set &Sigma, int q, const string &ord,
	 const localFct &local, int x, int y,
	 state_set touch, int card);

/* prints Lqca A in a compact, but readable representation.
 */
void print_compact(const Lqca &A, ostream &out=cout)
{
  static char *phase[4] = {"+1", "+i", "-1", "-i"};
  char        *sep = "";
  string       n;
  state        x;
  forall_defined(n,A.local) {
    if (sep[0] == '\0')
      sep = ". ";
    else {
      out << sep << n << "=";
      forall_nonzero(x, A.local[n] ) {
	out << phase[ int(4 + arg( A.local[n][x] ) * 2 / PI +0.5)%4 ];
	out << ' ' << x << ' ';
      }
    }
  }
  out << endl;
}

/* enumerates all Sigma, i.e. every cardinal q 
 */
void en1(void) {
  static string ord = 
    "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
  for (int q=1; q< ord.length(); q++) {
    state_set Sigma;
    for(int i=0; i<q ;i++)
      Sigma.insert(ord[i]);
    en2(Sigma, q, ord);
  }
}

/* Sets the local transition function for the quiescent state
 */
void en2(const state_set &Sigma, int q, const string &ord)
{
  localFct local;
  local[string(ord[0])+ord[0]] = ord[0];
  en3(Sigma, q, ord, local, 0, 0);
}

/* Sets the local transition function for the successor
   of the neighborhood (x,y)
   if (x,y) has no successor prints the automaton
   */
void en3(const state_set &Sigma, int q, const string &ord,
	 localFct local, int x, int y)
{
  if (x == q-1)
    if (y == q-1) {
      Lqca A(Sigma, ord[0], -1, 0, local); 
      if (wellformed(A)) {
	print_compact(A);
      }
      return;
    }
    else {
      y++;
      x=0;
    }
  else
    x++;
  en4(Sigma, q, ord, local, x,y, state_set(), 0);
}

/* Builds the touched state set 'touch' for the states 'next'..q-1
 */
void en4(const state_set &Sigma, int q, const string &ord,
	 localFct local, int x, int y,
	 state_set touch, int next)
{
  if (next == q) {
    if ( ! touch.empty() )
      en5(Sigma, q, ord, local, x,y, touch, touch.size());
  }
  else {
    en4(Sigma, q, ord, local, x,y, touch, next+1);
    touch.insert(ord[next]);
    en4(Sigma, q, ord, local, x,y, touch, next+1);
  }
}
   
/* enumerate all superpositions from (x,y) into touch
 */
void en5(const state_set &Sigma, int q, const string &ord,
	 const localFct &local, int x, int y,
	 state_set touch, int card)
{
  static Complex phase[4] = { Complex( 1,0), Complex(0, 1), 
			      Complex(-1,0), Complex(0,-1) };
  if ( touch.empty() ) 
    en3(Sigma, q, ord, local, x,y);
  else {
    state z = touch.choose();
    touch.del(z);
    for (int i=0; i<4; i++) {
      localFct plus(local);
      plus[string(ord[x])+ord[y]] += stateSupo(z, 1/sqrt(card) * phase[i]);
      en5(Sigma, q, ord, plus, x,y, touch, card);
    }
  }
}
       
// -------------------------- MAIN ----------------------------------
/* first argument: description file of the automaton
 */
const char *help ="\
Usage:  %s  file [options] [config [steps]]\n\
   options: -p(recision) epsilon   every x<epsilon is considered as 0\n\
            -a(automaton)          prints the automaton\n\
            -g(raphs)              prints both graphs\n\
            -w(ellformed)          don't verify if its well-formed\n\
            -m(matrix)   n         prints first n collumns/rows of the global matrix\n\
            -b(oldfaced)           prints in bold the cell indexed by 0\n\
            -i(mage)               prints the table as a bitmap (!=0 is black)\n\
            -e(numerate)           enumerate well-formed QLCA\n\
   default precision is %g\n\
   default number of steps is %u\n";

void main(int argc, char *argv[]) {
  Lqca A;
  int steps = 10;
  int i     = 2;  // argument index
  int n     = 0;  // nb of row/col to print (0 if none)
  if (argc == 1) {
    printf(help, argv[0], precision, steps);
    exit(1);
  }
                              // Scans options
  while (i<argc && argv[i][0]=='-') 
    switch (argv[i][1]) {
    case 'm':
      n = atoi(argv[i+1]);
      i += 2;
      break;
    case 'p':
      precision = atof(argv[i+1]);
      i += 2;
      break;
    case 'a':
      print_automaton = true; i++; break;
    case 'b':
      print_bold      = true; i++; break;
    case 'i':
      print_bitmap    = true; i++; break;
    case 'g':
      print_graphs    = true; i++; break;
    case 'w':
      verify          = false;i++; break;
    case 'e':
      en1();                  i++; break;
    default:
      cerr << "Bad options\n";
      exit(1);
    }
  if (i+1 < argc)
    steps = atoi(argv[i+1]);
  Load(A, argv[1]);
  if (n)
    PrintGlobal(A, n, cout);
  if ((!verify || wellformed(A)) && i<argc)
    simulate(A, argv[i], steps);
  exit(0);
}

