#!/usr/bin/env python3

# 2016 Christoph Durr

# Speed Scaling minimum Energy scheduling
# single machine
# Yao, Demers, Shenker
# complexity O(n^3)

from sys       import argv
# from copy      import *
from fractions import Fraction
from heapq import heappush, heappop

#constants
RELEASETIME = +1
DEADLINE = -1
alpha = 2

# ------------------------------------------------ instance

class Job:
    def __init__(self,r,d,w,j):
        self.r = r # release time
        self.d = d # deadline
        self.w = w # workload
        self.j = j # index in original instance (before sorting)
        self.E = 0 # energy
        self.C = 0 # completion time

    def __repr__(self):
        return "(r=%d,d=%d,w=%d,j=%d)"%(self.r, self.d, self.w, self.j)

    def __cmp__(self,other):
        return cmp(self.d,other.d)


def readInstance(inputstring):
    ''' J = all jobs, N = all job indices from 0 to n-1 '''
    global J, N, n
    jobs = []
    l = inputstring.split()
    # decode jobs
    for j in range(len(l)):
        try:
            r,d,w = map(int,l[j].split("/"))
        except ValueError:
            fatalError('Ill-formed job description "%s", '    \
                           'try removing unnecessary spaces'%l[j])
        if not (0<=r and r<d and w>0):
            fatalError('Ill-formed job description "%s", '    \
                           'release time must be before deadline, and weight positive'%l[j])
        jobs.append(Job(r,d,w,j))
    return jobs


# ------------------------------------------------ YDS schedule

def yds(jobs):
    events = []
    for i in range(len(jobs)):
        events.append((jobs[i].r, RELEASETIME, i))
        events.append((jobs[i].d, DEADLINE, i))
    events.sort()
    executedjobs = 0
    executed = [False] * len(jobs)
    todo = [job.w for job in jobs]
    schedule = []
    # print(events)

    while executedjobs < len(jobs):
        # determine the densest interval
        best = (float('-inf'), 0, 0, [])
        activejobs = 0
        for a in range(len(events)):
            now, aside, i = events[a]
            if executed[i]:
                activejobs += aside
            if aside == RELEASETIME and not executed[i]:     # for every release time
                availabletime = 0
                forcedwork = 0
                forcedjobs = []
                activesincea = activejobs
                for b in range(a + 1, len(events)):          # scan events
                    previous = now
                    now, bside, j = events[b]
                    if activesincea == 0:                    # maintain active jobs
                        availabletime += now - previous
                    if executed[j]:
                        activesincea += bside
                    elif bside == DEADLINE and jobs[j].r >= jobs[i].r:   # deadline is end of interval
                        forcedwork += jobs[j].w
                        forcedjobs.append(j)
                        interval = (Fraction(forcedwork, availabletime), a, b, set(forcedjobs))
                        if interval[0] > best[0]:
                            best = interval                  # densest interval
        # print(best)
        speed, a, b, forcedjobs = best
        pending = []
        activejobs = 0
        now = 0
        currentjob = None
        for c in range(b + 1):                               # scan interval
            previous = now
            now, side, j = events[c]
            if activejobs == 0 and c > a:                    # idle time to be filled
                while previous < now:
                    if currentjob is None or todo[currentjob] <= 0:  # earliest deadline first
                        currentjob = heappop(pending)[1]        # ignore deadline
                    work = todo[currentjob]
                    length = min(work / speed, now - previous)   # this job is executed for length
                    schedule.append((previous, previous + length, speed, currentjob))
                    previous += length
                    todo[currentjob] -= length * speed
                    if todo[currentjob] == 0:
                        jobs[currentjob].C = previous
                        jobs[currentjob].E = jobs[currentjob].w * speed ** (alpha - 1)
            if executed[j]:
                activejobs += side
            if j in forcedjobs and side == RELEASETIME:
                heappush(pending, (jobs[j].d, j))
                if currentjob is not None:
                    heappush(pending, (jobs[currentjob].d, currentjob))
                currentjob = heappop(pending)[1]
        for j in forcedjobs:
            executed[j] = True
        executedjobs += len(forcedjobs)
    # merge blocks of same size
    schedule.sort()
    clean = []
    for slot in schedule:
        start, end, speed, job = slot
        if clean and clean[-1][-1] == job:
            before = clean[-1][0]
            clean[-1] = (before, end, speed, job)
        else:
            clean.append(slot)
    return clean


# ------------------------------------------------ visual schedule


svgHeader = '<?xml version="1.0" standalone="no"?>\n'                     \
          '<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.0//EN" '               \
          '"http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd">\n' \
          '<svg xmlns="http://www.w3.org/2000/svg"'                       \
          ' font-family="sans-serif" font-size="10pt"'                    \
          ' xmlns:xlink="http://www.w3.org/1999/xlink" '                  \
          ' width="%d" height="%d" viewBox="0 0 %d %d">\n'                \
          '<text x="%d" y="%d" text-anchor="start">%s'                    \
          '</text>\n'

def schedule2svg(msg, jobs, S):
    # constants
    w = 18
    d =  3
    width  = w * max(100, max((job.d for job in jobs), default=0) + 2)
    base   = w * max((speed for (_,_,speed,_) in S), default=2)
    height = base + (len(jobs) + 3) * w + d

    print(svgHeader % (width, height, width, height, d, height - 2 * d, msg))

    # job release time deadline span with processing time
    for job in jobs:
        y1 = base+(job.j+2)*w
        x1 = job.r*w
        x2 = job.d*w
        y2 = y1-d
        L = [(x1,y2),(x1,y1),(x2,y1),(x2,y2)]
        for i in range(len(L)-1):
            print('<line x1="%d" y1="%d" x2="%d" y2="%d" '                    \
                  'style="stroke:grey"/>\n'%(L[i][0], L[i][1], L[i+1][0], L[i+1][1]))
        print('<text x="%d" y="%d" text-anchor="start">w%d=%d, C=%.2f, E=%.2f</text>'         \
               % (x1+d, y2, job.j+1, job.w, float(job.C), float(job.E)))

    # compute each job execution
    for (tstart,tend,speed,job) in S:
        if job !=-1:
            s = int(speed*w)
            y = base  - s
            x = tstart*w
            l = (tend-tstart)*w
            h = s
            print('<rect x="%d" y="%d" width="%d" height="%d" '       \
                  ' style="fill:yellow;stroke-width:1;stroke:rgb(0,0,0)" />'  \
                  % (x, y, l, h))
            if speed<1:
                y = base - s -d
            else:
                y = y + h - d
            print('<text x="%d" y="%d" text-anchor="start">%d</text>'     \
                  %(x + d, y, job+1))

    print('</svg>')


def fatalError(msg):
    print(svgHeader % (1000,50,1000,50,3, 45, "Fatal error: "+msg))
    print('</svg>')
    sys.exit(1)


def cost(S):
    total = 0
    for (tstart,tend,speed,job) in S:
        total += (tend - tstart) * float(speed) ** alpha
    return "total energy = %f" % total

# ------------------------------------------------ main program

if __name__=="__main__":
    if (len(argv) != 3):
        print('Usage: minEnergyScheduleYDS.py "alpha" "<list of release time/deadline/workload>"')
        exit(0)

    if '.' in argv[1]:
        alpha = float(argv[1])
    else:
        alpha = int(argv[1])
    jobs = readInstance(argv[2])
    schedule = yds(jobs)
    schedule2svg(cost(schedule), jobs, schedule)
